#include "sysinfo.h"

#define GETTEXT_PACKAGE "gtk-hello"

SysInfo::SysInfo()
{
    this->load_processors_info();
    this->load_memory_info();
    this->load_disk_info();
    this->load_uname_info();
    this->load_gnome_version();
}

SysInfo::~SysInfo() {}

/*!
 * Retrieve system's memory information
 */
void SysInfo::load_memory_info()
{
    glibtop_mem mem;

    glibtop_get_mem(&mem);
    this->memory_bytes = mem.total;
}

/*!
 * Retrieve system's processor(s) information
 */
void SysInfo::load_processors_info()
{
    const glibtop_sysinfo *info = glibtop_get_sysinfo();

    for (guint i = 0; i != info->ncpu; ++i) {
        const char * const keys[] = { "model name", "cpu" };
        gchar *model = 0;

        for (guint j = 0; !model && j != G_N_ELEMENTS(keys); ++j)
            model = static_cast<char*>(g_hash_table_lookup(info->cpuinfo[i].values,
                                                           keys[j]));

        if (!model)
            model = _("Unknown CPU model");

        this->processors.push_back(model);
    }
}

/*!
 * Retrieve system's hard disk information
 */
void SysInfo::load_disk_info()
{
    glibtop_mountentry *entries;
    glibtop_mountlist mountlist;

    entries = glibtop_get_mountlist(&mountlist, 0);

    this->free_space_bytes = 0;

    for (guint i = 0; i != mountlist.number; ++i)
    {

        if (string(entries[i].devname).find("/dev/") != 0)
            continue;

        if (string(entries[i].mountdir).find("/media/") == 0)
            continue;

        glibtop_fsusage usage;
        glibtop_get_fsusage(&usage, entries[i].mountdir);
        this->free_space_bytes += usage.bavail * usage.block_size;
    }

    g_free(entries);
}

/*!
 * Retrieve system's user name
 */
void SysInfo::load_uname_info()
{
    this->hostname = uname().nodename;
    this->kernel = string(uname().sysname) + ' ' + uname().release;
}

/*!
 * Retrieve current genome version being used on the system
 */
void SysInfo::load_gnome_version()
{
    QDomDocument doc("mydocument");
    QFile file("/usr/share/gnome-about/gnome-version.xml");
    if (!file.open(QIODevice::ReadOnly))
        return;
    if (!doc.setContent(&file))
    {
        file.close();
        return;
    }
    file.close();
    QString values[3];
    QXmlQuery query;
    QXmlQuery entries;



    const QString nodes[3] = { "platform",
                    "minor",
                    "micro" };

    QDomElement docElem = doc.documentElement();

    QDomNode n = docElem.firstChild();
    int i = 0;
    while(!n.isNull())
    {
        QDomElement e = n.toElement(); // try to convert the node to an element.
        if(!e.isNull())
        {
            if(i >= 3)
                break;
            if(nodes[i] == e.tagName())
            {
                values[i] = e.text();
            }
            i++;
            n = n.nextSibling();
        }
    }
    this->gnome_version = values[0] + '.' + values[1] + '.' + values[2];
}

 QString SysInfo::getHostName()
{
    return QString(hostname.c_str());
}

 QString SysInfo::getKernel()
 {
     return QString(kernel.c_str());
 }

 QString SysInfo::getGnome()
 {
     return gnome_version;
 }

QString SysInfo::convertGuintToQString(guint64 size, guint64 max_size, bool want_bits)
{
    enum
    {
        K_INDEX,
        M_INDEX,
        G_INDEX
    };

    struct Format
    {
        guint64 factor;
        const char* string;
    };

    const Format all_formats[2][3] =
    {
        { { 1UL << 10,	N_("%.1f KiB") },
        { 1UL << 20,	N_("%.1f MiB") },
        { 1UL << 30,	N_("%.1f GiB") } },
        { { 1000,	N_("%.1f kbit") },
        { 1000000,	N_("%.1f Mbit") },
        { 1000000000,	N_("%.1f Gbit") } }
    };

    const Format (&formats)[3] = all_formats[want_bits ? 1 : 0];

    if (want_bits)
    {
        size *= 8;
        max_size *= 8;
    }

    if (max_size == 0)
        max_size = size;

    if (max_size < formats[K_INDEX].factor)
    {
        const char *format = (want_bits
        ? dngettext(GETTEXT_PACKAGE, "%u bit", "%u bits", (guint) size)
        : dngettext(GETTEXT_PACKAGE, "%u byte", "%u bytes",(guint) size));
        return g_strdup_printf (format, (guint) size);
    }
    else
    {
        guint64 factor;
        const char* format = NULL;

        if (max_size < formats[M_INDEX].factor)
        {
            factor = formats[K_INDEX].factor;
            format = formats[K_INDEX].string;
        }
        else if (max_size < formats[G_INDEX].factor)
        {
        factor = formats[M_INDEX].factor;
        format = formats[M_INDEX].string;
        }
        else
        {
            factor = formats[G_INDEX].factor;
            format = formats[G_INDEX].string;
        }
        char *str = g_strdup_printf(_(format), this->free_space_bytes / (double)factor);
        return QString(str);
    }
   
}
QString SysInfo::getFreeSpace()
{
    return convertGuintToQString(this->free_space_bytes);
}

QString SysInfo::getProcessor()
{
    QString str = "", temp;
    int i = 0;
    vector<string>::iterator iter;
    for(iter = processors.begin(); iter != processors.end(); iter++)
    {
        string s = *iter;
        i++;
        temp = QString(s.c_str());
        str += "Processor ";
        str.append(QString::number(i));
        str.append(": ");
        str.append(temp);
        str.append("\n");
    }
    return str;
}
QString SysInfo::getMemory()
{
    return convertGuintToQString(this->memory_bytes);
}

std::string SysInfo::exec(char* cmd)
{
    FILE* pipe = popen(cmd, "r");
    if (!pipe) return "ERROR";
    char buffer[128];
    std::string result = "";
    while(!feof(pipe)) {
        if(fgets(buffer, 128, pipe) != NULL)
                result += buffer;
    }
    pclose(pipe);
    return result;
}

QString SysInfo::getRelease()
{
    std::string str = exec("lsb_release -irc");
    string::iterator it = str.begin();
    int pos, pos1;

    pos = str.find_first_of(':', 0);
    str.replace(it, it + pos + 2, "");

    pos = str.find_first_of('\n', 0);
    str.replace(it + pos, it + pos + 1, " (");

    pos++;
    pos1 = str.find_first_of(':', 0);
    str.replace(it + pos + 1, it + pos1 + 2, "");

    pos = str.find_first_of('\n', 0);
    str.replace(it + pos, it + pos + 1, " ");

    pos1 = str.find_first_of(':', 0);
    str.replace(it + pos + 1, it + pos1 + 2, "");

    pos = str.find_first_of('\n', 0);
    str.replace(it + pos, it + pos + 1, ")");


    return QString(str.c_str());
}
