#include <stdio.h>	//for printf(),
#include <unistd.h>	//for close(),
#include <stdlib.h>	//for exit(),
#include <strings.h>	//for bzero(),
#include <sys/socket.h>	//for socket(), inet_pton()
#include <sys/types.h>	//for socket(), inet_pton()
#include <arpa/inet.h>	//for htons(), inet_pton()
#include <netinet/in.h>	//for struct sockaddr_in,

#define BUFFER 4096

/**
  * Used to connect to tcp day time server and print current time.
  * The use must provide the i.p. address of tcp day tine server
  * through command line argument and then time is printed in 
  * human readable format. Note that it will work only for ipv4
  * servers.
  *
  * @returns zero to calling program on success, else 1.
  * @param argc number of command line arguments passed to the program
  *             including program name itself.
  * @param argv array of command line arguments.
  *
  */
int main(int argc, char *argv[])
{
    //used to store file descriptor of socket when opened.
    int socket_file_descriptor; 

    //used to store line from server temporarily
    char temp_line[BUFFER];

    //used to count number of characters read from socket.
    int character_read;

    //used to store address of ipv4 server 
    struct sockaddr_in server_address;

    //make sure user has given server i.p. address through command line argument
    if(argc != 2)
    {
	fprintf(stderr, "%s: usage: %s <server_ip_address>\n", argv[0], argv[0]);
	exit(1);
    }

    //try to open socket
    socket_file_descriptor = socket(AF_INET, SOCK_STREAM, 0);

    //if socket opening failed then print error message and quit
    if(socket_file_descriptor < 0)
    {
	fprintf(stderr, "%s: cannot open socket.\n", argv[0]);
	exit(1);
    }

    //initialize struct server_address to all zeros.
    bzero(&server_address, sizeof(server_address));

    //set the protocol used to be ipv4
    server_address.sin_family = AF_INET;

    //set the port to be 13, by default tcp day time servers
    //listen on port 13. htons is used to convert byte ordering from
    //system dependent (bigendian|littlendian) to network type, i.e.
    //system indepent. htons is host to network short.
    server_address.sin_port = htons(13);

    //convert the given server i.p. address in string or ascii notation
    //to proper address in binary format. if conversion is not done
    //then print proper error message and exit. The function inet_pton
    //can convert ipv4(AF_INET) and ipv6(AF_INET6) addresses into their 
    //proper format. This is also the first argument. The second time takes
    //string to convert to binary format and third argument takes address 
    //where the converted value has to be placed if conversion is successfull.
    if(inet_pton(AF_INET, argv[1], &server_address.sin_addr) <= 0)
    {
	fprintf(stderr, "%s: the supplied ipv4 address %s is incorrect.\n", argv[0], argv[1]);
	exit(1);
    }

    //try to connect to server port using socket. The function socket
    //take three arguments. first of them  is  socket to be used to connect 
    //to server. Second is address of server_address structure type casted as
    //(struct sockaddr *). Third is sizeof() server_address structure. This size is important 
    //as we send the address of struct so connect function does not knows what 
    //is the length of structure. Also, since same connect is used to connect 
    //to ipv6 servers too, connect() internally cannot do sizeof(struct sockaddr_in)
    //to find the size of socket.
    if(connect(socket_file_descriptor, (struct sockaddr *) &server_address, sizeof(server_address)) < 0)
    {
	fprintf(stderr, "%s: cannot connect to server %s\n", argv[0], argv[1]);
	exit(1);
    }

    //read up to BUFFER-1 bytes from server. i.e. read anything from 0 to BUFFER -1
    //bytes as written by server in socket. it returns number of characters read.
    //It can be zero too in case server has not written anything.
    character_read = read(socket_file_descriptor, temp_line, BUFFER-1);

    //null terminate the current string
    temp_line[character_read]='\0';
        

    //to keep reading BUFFER-1 bytes till server is writing to socket.
    while(character_read >0)
    {
	printf("%s",temp_line);
	character_read = read(socket_file_descriptor, temp_line, BUFFER-1);
	temp_line[character_read]='\0';
    }

    if(character_read < 0)
    {
	printf("Read error\n");
	exit(1);
    }

    //close the socket
    close(socket_file_descriptor);

    printf("\n");

    return 0;
}
